function [stk,range] = stack(varargin);
% Function that takes a stack of grayscale images and create a 3D matrix of
% the stack.
% Syntax: stk = stack('path','','name','',type,'','series',[])
% Input:    path = the path of the file, example: c:\data\november\
%           name = the name of the file, example: image
%           type = the type file, example: .tif
%           series = the range of the image series, example: [1 100]
%           NOTE: the above file series would be:
%           c:\data\november\image1-100.tif
%
% Output:   stk = matrix of images
%           range = the min and max scalar from each image

[path,name,type,series] = parse(varargin);

%get file info
try
    file = [path,name,num2str(series(1,1)),type];
    info = imfinfo(file);
    process = 0;
catch
    file = [path,name,type];
    info = imfinfo(file);
    process = 1;
end

%Initiation and setup
layer = 1;                          %Initiate the layer counter
size_x = info(1).Width;
size_y = info(1).Height;

%Initiate the stack matrixes
stk = zeros(size_x,size_y,series(1,2)-series(1,1)+1,'uint8');        %Initiated uint8 matrix, for imshow compatability
range = zeros(1,2,series(1,2)-series(1,1)+1,'uint8');            %Initiate the range array.  Remember change to double array for double accuracy.

%Process the images into our stack format
switch process
    case 1    %Multi image file: Grayscale, probably .tiff
        for i = series(1,1):series(1,2)                %Cycle through the files
            image = imread(file,i);     %Read in the image file
            image = image';                                 %Transpose the image array, because pixel dimensions are flipped
            stk(:,:,layer) = image;  %Write into stack matrix
            range(1,1,layer) = min(min(image));                 %Get the min value in the image
            range(1,2,layer) = max(max(image));                 %Get the max value in the image
            layer = layer+1;                                %Tick the layer counter by 1
        end
    otherwise
        for i = series(1,1):series(1,2)                %Cycle through the files
            image = imread([path,name,num2str(i),type]);     %Read in the image file
            image = image';                                 %Transpose the image array, because pixel dimensions are flipped
            stk(:,:,layer) = image;  %Write into stack matrix
            range(1,1,layer) = min(min(image));                 %Get the min value in the image
            range(1,2,layer) = max(max(image));                 %Get the max value in the image
            layer = layer+1;                                %Tick the layer counter by 1
        end
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [path,name,type,series] = parse(input)
%set defaults
path = [];
name = [];
type = [];
series = 1;

%Parse the input
if ~isempty(input)
    a = size(input,2);
    for i = 1:2:a
        value = input{1,i+1};
        switch input{1,i}
            case 'path'
                if ischar(value)
                    path = value;
                else
                    error(['Error: You must enter the a string for the directory path']);
                end
            case 'name'
                if ischar(value)
                    name = value;
                else
                    error(['Error: You must enter a string for the file name']);
                end
            case 'type'
                if ischar(value)
                    type = value;
                else
                    error(['Error: You must enter a string for the file type']);
                end
            case 'series'
                if isnumeric(value) & size(value,2)==2
                    series = value;
                else
                    error(['Error: You must enter two values in form [1 2] for the series designation']);
                end
            otherwise
                error(['Error: You entered jibberish.']);
        end
    end
end

%Checks and Balances
if isempty('type')
    type = response_box('title','You must enter the file type','input','.tif','caption','Examples: .tif, .jpg, .gif, etc.','position','center');
end
if isempty('series')
    series = yes_no_box('title','Do you want to continue?','caption1','You did not enter a series',...
        'caption2',['Currently your file name will be ',path,name,num2str(series),type],'position','center');
    if series == 0
        error('You might want to check help for syntax.  A stack of 1 image is kind of pointless.');
    end
end